document.addEventListener('DOMContentLoaded', initializePopup);

// Variável global para armazenar o estado atual da aba
let currentTab = 'active';

function initializePopup() {
    // 1. Configura os ouvintes de evento para as guias (tabs)
    document.getElementById('tab-active').addEventListener('click', () => switchTab('active'));
    document.getElementById('tab-completed').addEventListener('click', () => switchTab('completed'));

    // 2. Carrega a lista inicial de downloads ativos
    loadDownloads();
    
    // 3. Adiciona um listener para atualizar a lista em tempo real
    // Isso é importante, pois o download pode mudar de status enquanto o pop-up está aberto.
    chrome.downloads.onChanged.addListener(handleDownloadChange);
}

// Alterna entre as guias "Ativos" e "Concluídos"
function switchTab(tabName) {
    if (currentTab === tabName) return;

    currentTab = tabName;

    // Atualiza a classe 'active' nos botões
    document.getElementById('tab-active').classList.toggle('active', tabName === 'active');
    document.getElementById('tab-completed').classList.toggle('active', tabName === 'completed');

    loadDownloads();
}

// Função principal para carregar e exibir os downloads
function loadDownloads() {
    const listElement = document.getElementById('downloads-list');
    listElement.innerHTML = ''; // Limpa a lista atual

    const query = {
        orderBy: ['-startTime'], // Ordem: os mais recentes primeiro
        limit: 20 // Limita o número de itens para melhor performance
    };

    // Define o filtro de estado baseado na aba ativa
    if (currentTab === 'active') {
        query.state = 'in_progress';
    } else {
        query.state = 'complete';
        // Filtro para garantir que apenas downloads que não foram removidos do histórico sejam exibidos
        query.exists = true; 
    }

    // Busca os downloads usando a API do Chrome
    chrome.downloads.search(query, (downloads) => {
        
        const filteredDownloads = downloads.filter(download => {
            // Verifica o estado (in_progress ou complete)
            const matchesState = (currentTab === 'active' && download.state === 'in_progress') || 
                                 (currentTab === 'completed' && download.state === 'complete');
            
            // Verifica a existência (apenas downloads completos devem ser filtrados por 'exists')
            const matchesExistence = (currentTab === 'active' || download.exists === true);

            return matchesState && matchesExistence;
        });

        if (filteredDownloads.length === 0) {
            // Exibe a mensagem de estado vazio
            listElement.innerHTML = `<div class="empty-state"><p id="empty-message">Nenhum download ${currentTab === 'active' ? 'ativo no momento.' : 'concluído recentemente.'}</p></div>`;
            return;
        }

        filteredDownloads.forEach(download => {
            const itemHTML = createDownloadItemHTML(download);
            listElement.insertAdjacentHTML('beforeend', itemHTML);
        });

        // Configura os listeners dos botões após a criação dos itens
        setupActionListeners(filteredDownloads);
    });
}

// Cria o HTML para um único item de download
function createDownloadItemHTML(download) {
    // Extrai apenas o nome do arquivo, removendo caminhos
    const fileName = download.filename.split('/').pop().split('\\').pop(); 
    const isComplete = download.state === 'complete';
    const isPaused = download.state === 'in_progress' && download.paused;
    // Calcula o progresso, tratando 0/0 como 0 para evitar NaN
    const progress = isComplete ? 100 : (download.bytesReceived / download.totalBytes) * 100 || 0;
    
    let statusText = '';
    let actionsHTML = '';
    let icon = '❓'; // Ícone padrão
    
    // --- LÓGICA MELHORADA DOS ÍCONES ---
    if (fileName.endsWith('.exe')) icon = '💻';
    else if (fileName.endsWith('.jar')) icon = '☕';
    else if (fileName.endsWith('.py')) icon = '🐍';
    else if (fileName.endsWith('.pdf')) icon = '📄';
    else if (fileName.match(/\.(jpg|jpeg|png|gif|webp|svg)$/i)) icon = '🖼️';
    else if (fileName.match(/\.(mp4|mkv|avi|mov|webm)$/i)) icon = '🎬';
    else if (fileName.match(/\.(mp3|wav|ogg|flac)$/i)) icon = '🎶';
    else if (fileName.match(/\.(doc|docx|odt)$/i)) icon = '📰';
    else if (fileName.match(/\.(xls|xlsx|csv|ods)$/i)) icon = '📊';
    else if (fileName.match(/\.(zip|rar|7z|tar|gz)$/i)) icon = '📦';
    else if (fileName.match(/\.(js|html|css|php|json)$/i)) icon = '💾';
    else if (fileName.endsWith('.txt')) icon = '🗒️';
    else icon = '📁';
    
    if (isComplete) {
        // Downloads Concluídos
        const sizeMB = (download.totalBytes / 1048576).toFixed(2);
        const endTime = new Date(download.endTime).toLocaleTimeString('pt-BR', {hour: '2-digit', minute:'2-digit'});
        statusText = `${sizeMB} MB - Concluído às ${endTime}`;
        
        // APENAS O BOTÃO "Abrir Pasta" RESTOU AQUI
        actionsHTML = `
            <button class="action-button primary" data-id="${download.id}" data-action="open-folder">📂 Abrir Pasta</button>
        `;
    } else {
        // Downloads Ativos / Em Pausa
        const sizeMB = (download.bytesReceived / 1048576).toFixed(2);
        
        if (isPaused) {
            statusText = `PAUSADO - ${sizeMB} MB`;
            actionsHTML = `
                <button class="action-button primary" data-id="${download.id}" data-action="resume">▶️ Retomar</button>
                <button class="action-button secondary" data-id="${download.id}" data-action="cancel">❌ Cancelar</button>
            `;
        } else {
            // Exibe a porcentagem e tamanho atual
            statusText = `${progress.toFixed(1)}% (${sizeMB} MB)`;
            actionsHTML = `
                <button class="action-button primary" data-id="${download.id}" data-action="pause">⏸️ Pausar</button>
                <button class="action-button secondary" data-id="${download.id}" data-action="cancel">❌ Cancelar</button>
            `;
        }
    }

    return `
        <div class="download-item" data-id="${download.id}" data-state="${download.state}">
            <div class="file-info">
                <span class="file-icon">${icon}</span>
                <div class="file-details">
                    <div class="file-name" title="${fileName}">${fileName}</div>
                    <div class="progress-bar-container" ${isComplete ? 'style="display: none;"' : ''}>
                        <div class="progress-bar" style="width: ${progress}%;"></div>
                    </div>
                    <div class="status-text">${statusText}</div>
                </div>
            </div>
            <div class="actions">${actionsHTML}</div>
        </div>
    `;
}

// Configura os listeners para os botões de ação
function setupActionListeners(downloads) {
    document.querySelectorAll('.action-button').forEach(button => {
        button.addEventListener('click', (e) => {
            const downloadId = parseInt(e.target.dataset.id);
            const action = e.target.dataset.action;

            if (action === 'pause') {
                chrome.downloads.pause(downloadId, loadDownloads);
            } else if (action === 'resume') {
                chrome.downloads.resume(downloadId, loadDownloads);
            } else if (action === 'cancel') {
                chrome.downloads.cancel(downloadId, loadDownloads);
            } else if (action === 'open-folder') {
                chrome.downloads.show(downloadId); // Abre a pasta contendo o arquivo
            } 
            // A AÇÃO 'remove-list' FOI REMOVIDA DESTE BLOCO
        });
    });
}

// Lida com as mudanças de estado do download em tempo real
function handleDownloadChange(downloadDelta) {
    // Se o pop-up estiver aberto, recarrega a lista para mostrar o novo estado.
    if (downloadDelta.state || downloadDelta.paused || downloadDelta.bytesReceived) {
        // Pequeno timeout para evitar recarregamentos muito rápidos e instabilidade visual
        setTimeout(loadDownloads, 100); 
    }
}